"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.defineSystemRoutes = defineSystemRoutes;
exports.importTemplateHandler = importTemplateHandler;
exports.licenseHandler = licenseHandler;
exports.systemInfoHandler = systemInfoHandler;
var _stream = require("stream");
var _constants = require("../../../utils/constants");
var _configSchema = require("@kbn/config-schema");
/*
 *    Copyright 2020 floragunn GmbH
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

function systemInfoHandler({
  searchGuardBackend,
  logger
}) {
  return async function (context, request, response) {
    try {
      const body = await searchGuardBackend.systeminfo(request.headers);

      // Avoid Cacheable SSL Page
      // https://floragunn.atlassian.net/browse/ITT-2388
      return response.ok({
        body,
        headers: {
          'Cache-Control': 'no-cache, no-store'
        }
      });
    } catch (error) {
      logger.error(`systemInfoHandler: ${error.stack}`);
      return response.internalError({
        body: error
      });
    }
  };
}
function licenseHandler({
  searchGuardBackend,
  logger
}) {
  return async function (context, request, response) {
    try {
      const body = await searchGuardBackend.uploadLicense(request.headers, request.body);
      return response.ok({
        body
      });
    } catch (error) {
      logger.error(`licenseHandler: ${error.stack}`);
      // TODO: We need to consolidate the error response formats
      if (error && error.statusCode && error.statusCode === 400) {
        let message = "License invalid";
        if (error.message) {
          try {
            const parsedMessage = JSON.parse(error.message);
            if (parsedMessage && parsedMessage.message) {
              message = parsedMessage.message;
            }
          } catch (error) {
            // Ignore, probably unknown response format
          }
        }
        return response.customError({
          body: {
            message: 'License invalid',
            attributes: {
              body: message
            }
          },
          statusCode: 400
        });
      }
      return response.internalError({
        body: error
      });
    }
  };
}
function defineSystemRoutes({
  searchGuardBackend,
  logger,
  kibanaCore
}) {
  const router = kibanaCore.http.createRouter();
  router.get({
    path: `${_constants.API_ROOT}/systeminfo`,
    validate: false,
    options: {
      authRequired: false
    }
  }, systemInfoHandler({
    searchGuardBackend,
    logger
  }));
  router.post({
    path: `${_constants.API_ROOT}/license`,
    validate: {
      body: _configSchema.schema.object({
        sg_license: _configSchema.schema.string()
      })
    },
    options: {
      authRequired: true
    }
  }, licenseHandler({
    searchGuardBackend,
    logger
  }));
  router.post({
    path: `${_constants.API_ROOT}/templates/auditlog`,
    validate: {
      body: _configSchema.schema.object({
        overwrite: _configSchema.schema.boolean()
      })
    },
    options: {
      authRequired: true
    }
  }, importTemplateHandler({
    searchGuardBackend,
    logger
  }));
}
function importTemplateHandler({
  searchGuardBackend,
  logger
}) {
  return async function (context, request, response) {
    try {
      const overwrite = request.body.overwrite === true ? true : false;
      const sgContext = await context.searchGuard;
      const savedObjectsClient = await sgContext.startContext.savedObjects.getScopedClient(request);
      const savedObjectsImporter = await sgContext.startContext.savedObjects.createImporter(savedObjectsClient);
      // Require the actual objects
      const data = require('./templates/template_auditlog.js');
      let JSONReadStream = _stream.Readable.from(data.template_auditlog);
      const importResponse = await savedObjectsImporter.import({
        readStream: JSONReadStream,
        overwrite: overwrite,
        compatibilityMode: true
      });
      return response.ok({
        body: importResponse
      });
    } catch (error) {
      logger.error(`importTemplateHandler: ${error.stack}`);
      return response.internalError({
        body: error
      });
    }
  };
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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